<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Auth;

class RoleController extends Controller
{
    /*public function __construct()
    {
        $this->middleware('can:admin.roles.index');
    }*/

    public function index()
    {
        $roles = Role::all();
        return view('admin.roles.index', compact('roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $permissions = Permission::all(); // Obtén todos los permisos
        $structuredPermissions = $this->extractStructuredPermissions(config('adminlte.menu')); // Estructura jerárquica de permisos

        // Obtiene los permisos seleccionados para el rol actual, si es una edición
        $selectedPermissions = [];

        return view('admin.roles.create', compact('structuredPermissions', 'selectedPermissions'));
    }


    public function show(Role $role)
{
    // Obtener todos los permisos estructurados
    $structuredPermissions = $this->extractStructuredPermissions(config('adminlte.menu'));

    return view('admin.roles.show', compact('role', 'structuredPermissions'));
}


    /**
     * Show the form for editing the specified resource.
     */


    public function edit(Role $role)
    {
        // Asegúrate de usar el guard 'web'
        if (Auth::guard('web')->check()) {
            $permissions = Permission::all();
            $menu = config('adminlte.menu');
            $structuredPermissions = $this->extractStructuredPermissions($menu);
            $selectedPermissions = $role->permissions->pluck('name')->toArray();
            return view('admin.roles.edit', compact('role', 'structuredPermissions', 'selectedPermissions'));
        }

        return redirect()->route('dashboard')->with('error', 'No autorizado.');
    }



    public function update(Request $request, Role $role)
    {
        // Validación del nombre del rol
        $request->validate([
            'name' => "required|unique:roles,name,$role->id",
        ]);

        // Actualización del rol
        $role->update($request->all());

        // Obtener los IDs de los permisos seleccionados
        $permissionIds = Permission::whereIn('name', $request->permissions)->pluck('id')->toArray();

        // Sincronizar los permisos con el rol utilizando los IDs
        $role->permissions()->sync($permissionIds);

        return redirect()->route('admin.roles.show', $role)->with('success', 'update');
    }





    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:roles',
        ]);
        $role = Role::create($request->all());

        // Asegúrate de obtener el ID del permiso y sincronizarlo
        $permissionIds = Permission::whereIn('name', $request->permissions)->pluck('id')->toArray();
        $role->permissions()->sync($permissionIds);

        return redirect()->route('admin.roles.show', $role)->with('success', 'store');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role)
    {
        $role->delete();
        return redirect()->route('admin.roles.index')->with('success', 'El rol se eliminó con éxito');
    }

    public function generatePermissions()
    {
        $menu = config('adminlte.menu');

        // Extraer permisos y generar estructura jerárquica
        $structuredPermissions = $this->extractStructuredPermissions($menu);

        // Crear permisos en la base de datos
        $this->createPermissions($structuredPermissions);

        return view('admin.roles.generate-permissions', compact('structuredPermissions'));
    }

    private function extractStructuredPermissions(array $menu)
    {
        $structuredPermissions = [];

        foreach ($menu as $item) {
            $entry = [
                'text' => $item['text'],
                'icon' => $item['icon'] ?? null,
                'route' => $item['route'] ?? null,
                'submenu' => !empty($item['submenu']) ? $this->extractStructuredPermissions($item['submenu']) : [], // Manejo recursivo
            ];

            $structuredPermissions[] = $entry;
        }

        return $structuredPermissions;
    }


    private function createPermissions(array $structuredPermissions)
    {
        foreach ($structuredPermissions as $item) {
            $this->createPermissionIfNotExists($item);

            // Procesar submenús recursivamente
            if (!empty($item['submenu'])) {
                $this->createPermissions($item['submenu']);
            }
        }
    }

    private function createPermissionIfNotExists(array $item)
    {
        if (isset($item['route'])) {
            // Generar descripción
            $description = $this->generateDescription($item['route']);

            // Verificar si el permiso ya existe
            $existingPermission = Permission::where('name', $item['route'])->first();
            if (!$existingPermission) {
                Permission::create([
                    'name' => $item['route'],
                    'guard_name' => 'web',
                    'description' => $description,
                ]);
            }
        }
    }

    private function generateDescription($route)
    {
        // Quitar prefijos y sufijos comunes
        $baseName = str_replace(['admin.', '.index', '.store', '.update', '.delete'], '', $route);

        // Convertir de snake_case o dot.notation a formato legible
        $description = ucwords(str_replace(['.', '_'], ' ', $baseName));

        // Retornar descripción generada
        return $description;
    }



    private function extractPermissions(array $menu)
    {
        $permissions = [];

        foreach ($menu as $item) {
            if (isset($item['route'])) {
                $permissions[] = $item['route'];
            }

            if (isset($item['submenu'])) {
                $permissions = array_merge($permissions, $this->extractPermissions($item['submenu']));
            }
        }

        return $permissions;
    }
}
